<?php
/**
 * eForm myCRED Hook Class
 * The class responsible for hooking eForm into myCRED
 * and awarding points based on submissions
 *
 * @author Swashata <swashata4u@gmail.com>
 * @package myCRED integration for eForm
 * @subpackage Hook
 */
if ( ! class_exists( 'IPT_FSQM_MC_Hook' ) ) :

	class IPT_FSQM_MC_Hook extends myCRED_Hook {
		/**
		 * Construct
		 */
		function __construct( $hook_prefs, $type ) {
			parent::__construct( array(
				'id'       => 'fsqm',
				'defaults' => array(
					'log'     => __( '%plural% for submitting form: %name%', 'ipt_fsqm_mc' ),
					'limit_by'   => 'none',
					'limit_day' => '1',
				),
			), $hook_prefs, $type );
		}

		/**
		 * Hook into WordPress
		 */
		public function run() {
			// add during form submission
			add_action( 'ipt_fsqm_hook_save_insert',  array( $this, 'fsqm_submit' ), 10, 1 );

			// Add custom tags filter
			add_filter( 'mycred_parse_log_entry_fsqm', array( $this, 'parse_custom_tags' ), 10, 2 );
		}

		/**
		 * Check if the user qualifies for points
		 */
		public function fsqm_submit( $data ) {
			// do not do anything if user isn't logged in
			if ( ! is_user_logged_in() ) {
				return;
			}

			// do not do anything if form does not have settings enabled
			if ( ! isset( $data->settings['integration']['mycred'] ) || true != $data->settings['integration']['mycred']['enabled'] ) {
				return;
			}

			// do not do anything if form settings exclude current myCRED type
			if ( isset( $data->settings['integration']['mycred']['disabled_pt'] ) && in_array( $this->mycred_type, (array) $data->settings['integration']['mycred']['disabled_pt'] ) ) {
				return;
			}

			// get the user id
			$user_id = wp_get_current_user()->ID;
			// Check if user is excluded (required)
			if ( $this->core->exclude_user( $user_id ) ) return;

			// Prepare the ref id
			$ref_id = $data->data_id;

			// Override the limit if set
			if ( isset( $data->settings['integration']['mycred']['limit_by'] ) && $data->settings['integration']['mycred']['limit_by'] != 'no' ) {
				$this->prefs['limit_by'] = $data->settings['integration']['mycred']['limit_by'];
			}
			if ( isset( $data->settings['integration']['mycred']['limit_day'] ) && $data->settings['integration']['mycred']['limit_day'] != '' ) {
				$this->prefs['limit_day'] = $data->settings['integration']['mycred']['limit_day'];
			}

			// Prepare the data
			$mdata = array(
				'form_id' => "$data->form_id",
				'referer' => $data->data->referer,
				'name' => $data->name,
			);

			// Make sure this is a unique event
			// This will also check for limits
			if ( $this->has_entry( 'fsqm', $ref_id, $user_id, $mdata ) ) {
				return;
			}


			// Prepare the amount
			$creds = 0;
			switch ( $data->settings['integration']['mycred']['point_type'] ) {
				default :
				case 'static' :
					$creds = $data->settings['integration']['mycred']['static_point'];
					break;
				case 'score' :
					$creds = $data->data->score * (float) $data->settings['integration']['mycred']['multiplier'];
					break;
				case 'mathematical' :
					$creds = $data->data->freetype[$data->settings['integration']['mycred']['mid']]['value'] * (float) $data->settings['integration']['mycred']['multiplier'];
					break;
			}

			// Stop if negative rewards are disabled
			if ( $creds < 0 && $data->settings['integration']['mycred']['disable_neg'] == true ) {
				return;
			}

			// Prepare the log
			$log = $this->prefs['log'];

			// Execute
			$this->core->add_creds(
				'fsqm',
				$user_id,
				$creds,
				$log,
				$ref_id,
				$mdata,
				$this->mycred_type
			);
		}

		/**
		 * Parse Custom Tags in Log
		 * @since 1.1
		 * @version 1.1.1
		 */
		public function parse_custom_tags( $content, $log_entry ) {
			global $wpdb, $ipt_fsqm_info;

			$data = maybe_unserialize( $log_entry->data );

			$category = $wpdb->get_var( $wpdb->prepare( "SELECT c.name name FROM {$ipt_fsqm_info['category_table']} c LEFT JOIN {$ipt_fsqm_info['form_table']} f ON f.category = c.id WHERE f.id = %d", $data['form_id'] ) );

			$content = str_replace( array( '%name%', '%id%', '%from%', '%formcat%' ), array( $data['name'], $data['form_id'], $data['referer'], $category ), $content );

			return $content;
		}

		/**
		 * Customize Limit Options
		 * @since 1.1
		 * @version 1.0
		 */
		public function custom_limit() {
			return array(
				'none' => __( 'No limits', 'ipt_fsqm_mc' ),
				'form'  => __( 'Once for each form', 'ipt_fsqm_mc' ),
				'day'   => __( 'Once for every n days', 'ipt_fsqm_mc' ),
				'formday' => __( 'Once for each form per n days', 'ipt_fsqm_mc' ),
			);
		}

		/**
		 * Custom Has Entry Check
		 * @since 1.1
		 * @version 1.1.1
		 */
		public function has_entry( $action = '', $reference = '', $user_id = '', $data = '', $type = '' ) {
			if ( $type == '' )
				$type = $this->mycred_type;

			global $wpdb;

			// Prepare some comparisons variables
			// s:7:"form_id";s:2:"58";
			$string = '%s:7:"form_id";s:' . strlen( $data['form_id'] ) . ':"' . $data['form_id'] . '";%';
			$current_time = date_i18n( 'U' );
			$accepted_time = $current_time - ( 86400 * abs( $this->prefs['limit_day'] ) ); // Subtract n day

			// If no limit
			if ( $this->prefs['limit_by'] == 'none' ) {
				return parent::has_entry( $action, $reference, $user_id, $data, $type );
			}

			// If limit by once for every form
			if ( $this->prefs['limit_by'] == 'form' ) {
				$sql = "SELECT id FROM {$this->core->log_table} WHERE ref = %s AND user_id = %d AND data LIKE %s AND ctype = %s";
				if ( $wpdb->get_var( $wpdb->prepare( $sql, $action, $user_id, $string, $type ) ) ) {
					return true;
				} else {
					return false;
				}
			}

			// If limit by once every form per day
			if ( $this->prefs['limit_by'] == 'formday' ) {
				$sql = "SELECT id FROM {$this->core->log_table} WHERE ref = %s AND user_id = %d AND data LIKE %s AND ctype = %s AND time >= %s";
				if ( $wpdb->get_var( $wpdb->prepare( $sql, $action, $user_id, $string, $type, $accepted_time ) ) ) {
					return true;
				} else {
					return false;
				}
			}

			// If limit by once every day (regardless of form id)
			if ( $this->prefs['limit_by'] == 'day' ) {
				$sql = "SELECT id FROM {$this->core->log_table} WHERE ref = %s AND user_id = %d AND ctype = %s AND time >= %s";
				if ( $wpdb->get_var( $wpdb->prepare( $sql, $action, $user_id, $type, $accepted_time ) ) ) {
					return true;
				} else {
					return false;
				}
			}

			return parent::has_entry( $action, $reference, $user_id, $data, $type );

		}

		/**
		 * Add Settings
		 */
		 public function preferences() {
			// Our settings are available under $this->prefs
			$prefs = $this->prefs;
			?>
	<!-- Some CSS -->
	<style type="text/css">
		#myCRED-wrap #accordion h4 .icon.fsqm {
			background: url('<?php echo plugins_url( '/static/images/hook-icon.png', IPT_FSQM_MC_Loader::$abs_file ); ?>') no-repeat scroll 0 -48px;
		}
		#myCRED-wrap #accordion h4 .icon.icon-active.fsqm {
			background-position: 0 0;
		}
	</style>
	<!-- Limits -->
	<label class="subheader"><?php _e( 'Limits', 'ipt_fsqm_mc' ); ?></label>
	<ol>
		<li>
	<?php
		add_filter( 'mycred_hook_impose_limits', array( $this, 'custom_limit' ) );
		$this->impose_limits_dropdown( 'limit_by', false );
		remove_filter( 'mycred_hook_impose_limits', array( $this, 'custom_limit' ) );
	?>
		</li>
		<li><strong><?php _e( 'Note!', 'ipt_fsqm_mc' ); ?></strong> <?php _e( 'It is recommended to set at least <strong>Once for each form</strong>. Otherwise users would get too much award point for every submissions.', 'ipt_fsqm_mc' ); ?></li>
	</ol>
	<label class="subheader"><?php _e( 'Days Limit', 'ipt_fsqm_mc' ); ?></label>
	<ol>
		<li>
			<div class="h2"><input type="number" name="<?php echo $this->field_name( 'limit_day' ); ?>" id="<?php echo $this->field_id( 'limit_day' ); ?>" value="<?php echo esc_attr( $prefs['limit_day'] ); ?>" size="8" min="0" /><br /><span class="description"><?php _e( 'If you choose limit by n days or limit by each form per n days then you can specify the number of days here. This can also be overriden from the form settings.', 'ipt_fsqm_mc' ); ?></span></div>
		</li>
	</ol>
	<!-- Then the log template -->
	<label class="subheader"><?php _e( 'Log template', 'ipt_fsqm_mc' ); ?></label>
	<ol>
		<li>
			<div class="h2"><input type="text" name="<?php echo $this->field_name( 'log' ); ?>" id="<?php echo $this->field_id( 'log' ); ?>" value="<?php echo esc_attr( $prefs['log'] ); ?>" class="long" /><br /><span class="description"><?php _e( '%name% will be replaced by form name. %id% will be replaced by form ID. %from% will be replaced by the URL from where the submission has occurred, %formcat% will be replaced by the form category.', 'ipt_fsqm_mc' ); ?></span></div>
		</li>
	</ol>
	<?php
		}

		/**
		 * Sanitize Preferences
		 */
		public function sanitise_preferences( $data ) {
			$new_data = $data;

			// Apply defaults if any field is left empty
			$new_data['log'] = ( ! empty( $data['log'] ) ) ? sanitize_text_field( $data['log'] ) : $this->defaults['log']; //

			return $new_data;
		}
	}

endif;
