<?php

class IPT_FSQM_Form_Elements_SQL extends IPT_FSQM_Form_Elements_Data {

	protected $sql_id = null;

	protected $table_name = null;

	protected $blacklists = array();

	protected $errors = array();

	protected $value_class = null;

	/**
	 * CSV Multi Option Delimiter
	 * @var string
	 */
	protected $option_delimiter = '\n';

	/**
	 * CSV Multi Row Delimiter
	 * @var string
	 */
	protected $row_delimiter = '\n\n';

	/**
	 * CSV Multi Number Range Delimiter
	 * @var string
	 */
	protected $range_delimiter = '/';

	/**
	 * Multi entry delimiter
	 *
	 * @var        string
	 */
	protected $entry_delimiter = ':';

	/*==========================================================================
	 * A few setters for config
	 *========================================================================*/
	/**
	 * Set the Multi Option Delimiter
	 * @param string $delimiter The delimiter character/string
	 */
	public function set_option_delimiter( $delimiter = "\n" ) {
		$this->option_delimiter = $this->filter_delimiters( $delimiter );
	}

	/**
	 * Set the Multi Row Delimiter
	 * @param string $delimiter The delimiter character/string
	 */
	public function set_row_delimiter( $delimiter = "\n\n" ) {
		$this->row_delimiter = $this->filter_delimiters( $delimiter );
	}

	/**
	 * Set the Range Delimiter
	 * @param string $delimiter The range delimiter (Default '/')
	 */
	public function set_range_delimiter( $delimiter = '/' ) {
		$this->range_delimiter = $this->filter_delimiters( $delimiter );
	}

	public function set_entry_delimiter( $delimiter = ':' ) {
		$this->entry_delimiter = $this->filter_delimiters( $delimiter );
	}

	public function filter_delimiters( $delimiter ) {
		switch ( $delimiter ) {
			default:
				return $delimiter;
				break;
			case '\t':
				return "\t";
				break;
			case '\n':
				return "\n";
				break;
			case '\n\n':
				return "\n\n";
				break;
			case '\n\n\n':
				return "\n\n\n";
				break;
			case '\n\n\n\n':
				return "\n\n\n\n";
				break;
		}
	}

	/*==========================================================================
	 * Getters for some config
	 *========================================================================*/
	/**
	 * Gets the ID of the RAW database entry.
	 *
	 * @return string
	 */
	public function get_sql_id() {
		return $this->sql_id;
	}

	/**
	 * Gets the CSV Multi Option Delimiter.
	 *
	 * @return string
	 */
	public function get_option_delimiter() {
		return $this->option_delimiter;
	}

	/**
	 * Gets the CSV Multi Row Delimiter.
	 *
	 * @return string
	 */
	public function get_row_delimiter() {
		return $this->row_delimiter;
	}

	/**
	 * Gets the Range Delimiter.
	 *
	 * @return string
	 */
	public function get_range_delimiter() {
		return $this->range_delimiter;
	}

	/**
	 * Gets the entry delimiter.
	 *
	 * @return     string  The entry delimiter.
	 */
	public function get_entry_delimiter() {
		return $this->entry_delimiter;
	}

	/*==========================================================================
	 * Constructor
	 *========================================================================*/
	public function __construct( $data_id ) {
		parent::__construct( $data_id );

		// Further process if
		if ( $this->data_id != null && $this->settings['direct_exp']['enabled'] == true ) {
			global $wpdb;
			$this->table_name = $wpdb->prefix . 'fsqm_direct_' . $this->form_id;

			// Get the sql id
			$sql_id = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM {$this->table_name} WHERE data_id = %d AND ref = %d", $this->data_id, 0 ) );
			if ( $sql_id ) {
				$this->sql_id = $sql_id;
			}

			// Set delimiters
			$this->set_range_delimiter( $this->settings['direct_exp']['frd'] );
			$this->set_option_delimiter( $this->settings['direct_exp']['mod'] );
			$this->set_row_delimiter( $this->settings['direct_exp']['mrd'] );
			$this->set_entry_delimiter( $this->settings['direct_exp']['med'] );

			// Set blacklists
			$this->blacklists = explode(',', strtolower( $this->settings['direct_exp']['excludes'] ) );

			// Get a value class
			$this->value_class = new IPT_eForm_Form_Elements_Values( $this->data_id );
			// Set delimiters there
			$this->value_class->set_range_delimiter( $this->get_range_delimiter() );
			$this->value_class->set_option_delimiter( $this->get_option_delimiter() );
			$this->value_class->set_row_delimiter( $this->get_row_delimiter() );
			$this->value_class->set_entry_delimiter( $this->get_entry_delimiter() );
		}
	}


	/*==========================================================================
	 * Main Processing Functions
	 *========================================================================*/

	public function delete_export() {
		if ( null == $this->table_name ) {
			return;
		}
		global $wpdb;
		do_action( 'ipt_exp_hook_sql_deleted', $this );
		$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} WHERE data_id = %d", $this->data_id ) );
	}

	public function save_to_table( $context = null ) {
		// Check the settings and data availability
		if ( $this->data_id == null || $this->settings['direct_exp']['enabled'] == false ) {
			return;
		}

		global $wpdb;

		// Check if the table is present
		if ( $wpdb->get_var( "SHOW TABLES LIKE '{$this->table_name}'" ) != $this->table_name ) {
			return;
		}

		$table_data = array(
			'data_id' => $this->data_id,
			'updated' => $this->data->date,
			'ref' => '0',
		);
		$submission_data = $this->create_query();
		$table_data = array_merge( $table_data, $submission_data );

		// Create an entry
		if ( $this->sql_id == null ) {
			$wpdb->insert( $this->table_name, $table_data, '%s' );
		// Update existing
		} else {
			// Here we check if we are to keep old entries
			// But we do this only for user updates
			if ( null !== $context && true === $this->settings['direct_exp']['keep_old'] && true === $context->user_update ) {
				// We update the existing one with the new data
				// and copy over the existing one to a new row with a reference set
				// First we do the copy
				$existing_data = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE id = %d", $this->sql_id ), ARRAY_A );
				if ( null !== $existing_data ) {
					// Change some variables
					unset( $existing_data['id'] );
					$existing_data['data_id'] = null;
					$existing_data['ref'] = $this->sql_id;
					// We need to change the table data updated because the original plugin
					// does not do that, and right now we do not intend to change it
					$table_data['updated'] = current_time( 'mysql' );
					// Create a copy
					$wpdb->insert( $this->table_name, $existing_data, '%s' );
				}
				// Now outside the if statement the update occurs
			}
			$wpdb->update( $this->table_name, $table_data, array( 'id' => $this->sql_id ), '%s', '%d' );
		}

		// Now check for limitations
		$this->check_limitation();
	}

	public function check_limitation() {
		global $wpdb;
		// Check for total limitations
		if ( (int) $this->settings['direct_exp']['total_limit'] > 0 ) {
			$limit_total = (int) $this->settings['direct_exp']['total_limit'];
			$total_data = $wpdb->get_var( "SELECT COUNT(id) FROM {$this->table_name} WHERE ref = 0" );
			if ( $total_data > $limit_total ) {
				$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} WHERE updated <= ( SELECT updated FROM ( SELECT updated FROM {$this->table_name} ORDER BY updated DESC LIMIT 1 OFFSET %d ) foo ) AND ref = 0", $limit_total ) );
			}
		}

		// Check for past n days limitation
		if ( (int) $this->settings['direct_exp']['day_limit'] > 0 ) {
			$limit_day = (int) $this->settings['direct_exp']['day_limit'];
			$limit_date = date( 'Y-m-d 00:00:00', current_time( 'timestamp' ) - ( $limit_day * 24 * 60 * 60 ) );
			$wpdb->query( $wpdb->prepare( "DELETE FROM {$this->table_name} WHERE updated < %s AND ref = 0", $limit_date ) );
		}
	}

	public function create_query() {
		// Initiate the insert array
		$insert = array();

		// Loop through all elements
		foreach ( ( array ) $this->mcq as $e_key => $element ) {
			if ( in_array( 'm' . $e_key, (array) $this->blacklists ) ) {
				continue;
			}
			$column_name = 'mcq' . $e_key;
			$insert[ $column_name ] = $this->get_value( $element, $e_key );
		}
		foreach ( ( array ) $this->freetype as $e_key => $element ) {
			if ( in_array( 'f' . $e_key, (array) $this->blacklists ) ) {
				continue;
			}
			$column_name = 'freetype' . $e_key;
			$insert[ $column_name ] = $this->get_value( $element, $e_key );
		}
		foreach ( ( array ) $this->pinfo as $e_key => $element ) {
			if ( in_array( 'o' . $e_key, (array) $this->blacklists ) ) {
				continue;
			}
			$column_name = 'pinfo' . $e_key;
			$insert[ $column_name ] = $this->get_value( $element, $e_key );
		}

		return $insert;
	}

	public function get_value( $element, $e_key ) {
		$value = '';
		$submission = $this->get_submission_from_data( array(
			'm_type' => $element['m_type'],
			'key' => $e_key,
		) );

		// We pass the hard work to the value class
		$value = $this->value_class->get_value( $element['m_type'], $e_key, 'string' );

		// // Check for built ins first
		// if ( method_exists( $this, 'sql_' . $element['type'] ) ) {
		// 	return call_user_func( array( $this, 'sql_' . $element['type'] ), $element, $e_key, $submission );
		// }

		// // Not set
		// // Check if external
		// $definition = $this->get_element_definition( $element );
		// if ( isset( $definition['callback_sql'] ) && is_callable( $definition['callback_sql'] ) ) {
		// 	return call_user_func( $definition['callback_sql'], $element, $e_key, $submission, $this );
		// }

		return $value;
	}

	public function get_revisions() {
		global $wpdb;

		// Get all the entries
		$revisions = $wpdb->get_results( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE ref = %d ORDER BY id DESC", $this->sql_id ), ARRAY_A );
		// Get the main entry
		$current = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$this->table_name} WHERE id = %d", $this->sql_id ), ARRAY_A );

		// Now we simply return it in an associative array
		return array(
			'current' => $current,
			'revisions' => (array) $revisions,
		);
	}

	public function get_table_info() {
		global $wpdb;
		return $wpdb->get_results( "SHOW FULL COLUMNS FROM {$this->table_name}" );
	}

	/*==========================================================================
	 * CSV Calculator for Elements
	 *========================================================================*/
	// MCQs
	public function sql_checkbox( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_radio( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_select( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_thumbselect( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_slider( $element, $e_key, $submission ) {
		return $submission['value'];
	}

	public function sql_range( $element, $e_key, $submission ) {
		return $submission['values']['min'] . $this->get_range_delimiter() . $submission['values']['max'];
	}

	public function sql_grading( $element, $e_key, $submission ) {
		$return = array();

		foreach ( $element['settings']['options'] as $o_key => $option ) {
			if ( ! isset ( $submission['options'][ $o_key ] ) ) {
				continue;
			}

			// Can be range or single
			if ( true == $element['settings']['range'] ) {
				$return[] = $option['label'] . $this->get_option_delimiter() . $submission['options'][ $o_key ]['min'] . $this->get_range_delimiter() . $submission['options'][ $o_key ]['max'];
			} else {
				$return[] = $option['label'] . $this->get_option_delimiter() . $submission['options'][ $o_key ];
			}
		}

		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_smileyrating( $element, $e_key, $submission ) {
		$return = array();
		if ( isset( $element['settings']['labels'][$submission['option']] ) ) {
			$return[] = $element['settings']['labels'][$submission['option']];
		}
		if ( $submission['feedback'] != '' && $element['settings']['show_feedback'] == true ) {
			$return[] = $submission['feedback'];
		}
		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_starrating( $element, $e_key, $submission ) {
		return $this->sql_make_numerics( $element, $e_key, $submission );
	}

	public function sql_scalerating( $element, $e_key, $submission ) {
		return $this->sql_make_numerics( $element, $e_key, $submission );
	}

	public function sql_spinners( $element, $e_key, $submission ) {
		return $this->sql_make_numerics( $element, $e_key, $submission );
	}

	public function sql_matrix( $element, $e_key, $submission ) {
		$return = array();

		foreach ( $element['settings']['rows'] as $r_key => $row ) {
			if ( ! isset( $submission['rows'][$r_key] ) ) {
				continue;
			}

			$sub = array();
			$sub[] = $row;

			foreach ( $element['settings']['columns'] as $c_key => $col ) {
				if ( in_array( (string) $c_key, $submission['rows'][$r_key] ) ) {
					$sub[] = $col;
				}
			}

			$return[] = implode( $this->get_option_delimiter(), $sub );
		}

		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_matrix_dropdown( $element, $e_key, $submission ) {
		$return = array();

		foreach ( $element['settings']['rows'] as $r_key => $row ) {
			if ( ! isset( $submission['rows'][$r_key] ) ) {
				continue;
			}
			foreach ( $element['settings']['columns'] as $c_key => $col ) {
				$sub = array();
				$sub[] = $row;
				$sub[] = $col;
				if ( isset( $submission['rows'][$r_key][$c_key] ) && isset( $element['settings']['options'][(string) $submission['rows'][$r_key][$c_key]] ) ) {
					$sub[] = $element['settings']['options'][(string) $submission['rows'][$r_key][$c_key]]['label'];
				}
				$return[] = implode( $this->get_option_delimiter(), $sub );
			}
		}
		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_likedislike( $element, $e_key, $submission ) {
		if ( $submission['value'] == 'like' ) {
			return $element['settings']['like'];
		} else {
			return $element['settings']['dislike'];
		}
	}

	public function sql_toggle( $element, $e_key, $submission ) {
		if ( $submission['value'] == false ) {
			return $element['settings']['off'];
		}
		return $element['settings']['on'];
	}

	public function sql_sorting( $element, $e_key, $submission ) {
		return $this->sql_make_sorting( $element, $e_key, $submission );
	}

	// Feedbacks
	public function sql_feedback_large( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_feedback_small( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_upload( $element, $e_key, $submission ) {
		$uploader = new IPT_FSQM_Form_Elements_Uploader( $this->form_id, $e_key );
		$uploads = $uploader->get_uploads( $this->data_id );

		if ( ! empty( $uploads ) ) {
			$return = array();
			foreach ( $uploads as $upload ) {
				if ( $upload['guid'] == '' ) {
					continue;
				}
				$return[] = $upload['name'] . $this->get_option_delimiter() . '(' . $upload['guid'] . ')';
			}
			return implode( $this->get_row_delimiter(), $return );
		} else {
			return __( 'No files uploaded.', 'ipt_fsqm_exp' );
		}
	}

	public function sql_mathematical( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_feedback_matrix( $element, $e_key, $submission ) {
		$return = array();

		foreach ( $element['settings']['rows'] as $r_key => $row ) {
			if ( ! isset( $submission['rows'][$r_key] ) ) {
				continue;
			}
			foreach ( $element['settings']['columns'] as $c_key => $col ) {
				$sub = array();
				$sub[] = $row;
				$sub[] = $col;
				$sub[] = $submission['rows'][$r_key][$c_key];
				$return[] = implode( $this->get_option_delimiter(), $sub );
			}
		}
		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_signature( $element, $e_key, $submission ) {
		$image = $this->convert_jsignature_image( $submission['value'] );
		if ( $image == '' ) {
			return '';
		}
		return 'data:image/png;base64,' . $image;
	}

	public function sql_gps( $element, $e_key, $submission ) {
		$return = array();

		foreach ( array( 'location_name', 'lat', 'long' ) as $pos ) {
			if ( isset( $submission[$pos] ) ) {
				$return[] = $element['settings'][$pos . '_' . 'label'] . $this->get_option_delimiter() . $submission[$pos];
			}
		}

		return implode( $this->get_row_delimiter(), $return );
	}

	// Pinfos
	public function sql_payment( $element, $e_key, $submission ) {
		global $wpdb, $ipt_fsqm_info;

		$payment_db = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM {$ipt_fsqm_info['payment_table']} WHERE data_id = %d", $this->data_id ) );

		if ( is_null( $payment_db ) ) {
			return '';
		}

		$payment_amount = $submission['value'];
		$discount_amount = 0;
		if ( isset( $submission['couponval'] ) && $submission['couponval'] != '' ) {
			$payment_amount = $submission['couponval'];
			$discount_amount = $submission['value'] - $payment_amount;
		}

		$invoiceid = str_replace( '{id}', $payment_db->id, $this->settings['payment']['invoicenumber'] );
		if ( $invoiceid == '' ) {
			$invoiceid = $payment_db->id;
		}

		$payment_status = IPT_FSQM_Form_Elements_Static::ipt_fsqm_get_payment_status();

		$payment_modes = IPT_FSQM_Form_Elements_Static::ipt_fsqm_get_payment_gateways();

		$returns = array();
		// Date
		$returns[] = __( 'Date', 'ipt_fsqm_exp' ) . ' : ' . date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $payment_db->date ) );

		// Invoice ID
		$returns[] = __( 'Invoice ID', 'ipt_fsqm_exp' ) . ' : ' . $invoiceid;

		// Status
		$returns[] = __( 'Status', 'ipt_fsqm_exp' ) . ' : ' . @$payment_status[$payment_db->status];

		// TxnID
		$returns[] = __( 'Transaction ID', 'ipt_fsqm_exp' ) . ' : ' . $payment_db->txn;

		// Payment gateway
		$returns[] = __( 'Payment gateway', 'ipt_fsqm_exp' ) . ' : ' . @$payment_modes[$payment_db->mode];

		// Item Name
		$returns[] = __( 'Item name', 'ipt_fsqm_exp' ) . ' : ' . $this->settings['payment']['itemname'];

		// Item SKU
		if ( $this->settings['payment']['itemsku'] != '' ) {
			$returns[] = __( 'Item SKU', 'ipt_fsqm_exp' ) . ' : ' . $this->settings['payment']['itemsku'];
		}

		// Item description
		$returns[] = __( 'Description', 'ipt_fsqm_exp' ) . ' : ' . $this->settings['payment']['itemdescription'];

		// Item price
		$returns[] = __( 'Price', 'ipt_fsqm_exp' ) . ' : ' . $submission['value'];

		// Coupon
		if ( $submission['coupon'] != '' && $discount_amount > 0 ) {
			$returns[] = __( 'Coupon', 'ipt_fsqm_exp' ) . ' : ' . $submission['coupon'];
			$returns[] = __( 'Discount', 'ipt_fsqm_exp' ) . ' : ' . number_format_i18n( $discount_amount, 2 );;
		}

		// Total
		$returns[] = __( 'Total', 'ipt_fsqm_exp' ) . ' : ' . number_format_i18n( $payment_amount, 2 );

		return implode( $this->get_option_delimiter(), $returns );
	}

	public function sql_f_name( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_l_name( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_email( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_phone( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_p_name( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_p_email( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_p_phone( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_textinput( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_textarea( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_password( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_keypad( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_datetime( $element, $e_key, $submission ) {
		return $this->sql_make_text( $element, $e_key, $submission );
	}

	public function sql_p_checkbox( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_p_radio( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_p_select( $element, $e_key, $submission ) {
		return $this->sql_make_mcqs( $element, $e_key, $submission );
	}

	public function sql_s_checkbox( $element, $e_key, $submission ) {
		if ( $submission['value'] == false ) {
			return 0;
		}
		return 1;
	}

	public function sql_address( $element, $e_key, $submission ) {
		return implode( $this->get_row_delimiter(), $submission['values'] );
	}

	public function sql_p_sorting( $element, $e_key, $submission ) {
		return $this->sql_make_sorting( $element, $e_key, $submission );
	}

	/*==========================================================================
	 * Helpers for CSV Calculators
	 *========================================================================*/
	public function sql_make_mcqs( $element, $e_key, $submission ) {
		$return = array();
		if ( ! is_array( $submission['options'] ) ) {
			$submission['options'] = array();
		}
		foreach ( $element['settings']['options'] as $o_key => $op ) {
			if ( in_array( (string) $o_key, $submission['options'] ) ) {
				$return[] = $op['label'];
			}
		}
		if ( isset( $element['settings']['others'] ) && $element['settings']['others'] == true ) {
			if ( in_array( 'others', $submission['options'] ) ) {
				$return[] = $element['settings']['o_label'];
			}
		}

		$return = implode( $this->get_option_delimiter(), $return );
		if ( isset( $element['settings']['others'] ) && $element['settings']['others'] == true && in_array( 'others', $submission['options'] ) ) {
			$return .= $this->get_row_delimiter() . $submission['others'];
		}

		return $return;
	}

	public function sql_make_text( $element, $e_key, $submission ) {
		$return = str_replace( "\r", "", $submission['value'] );
		return str_replace( "\n\n" , "\n", $return );
	}

	public function sql_make_numerics( $element, $e_key, $submission ) {
		$return = array();

		foreach ( $element['settings']['options'] as $o_key => $option ) {
			if ( ! isset( $submission['options'][ $o_key ] ) ) {
				continue;
			}
			$label = $option;
			if ( is_array( $option ) && isset( $option['label'] ) ) {
				$label = $option['label'];
			}
			$return[] = $label . $this->get_option_delimiter() . $submission['options'][ $o_key ];
		}

		return implode( $this->get_row_delimiter(), $return );
	}

	public function sql_make_sorting( $element, $e_key, $submission ) {
		$return = array();
		foreach ( (array) $submission['order'] as $o_key ) {
			$return[] = $element['settings']['options'][$o_key]['label'];
		}

		return implode( $this->get_row_delimiter(), $return );
	}
}
